## Version 4.3.3
## Author Jim Stevens
## 5/1/23
## Analyse light response curves
## Actinic light intensity, assimilation 
## FvFM OR Fq'Fm' values

## Housekeeping

rm(list=ls()) # Clear R workspace

library(tidyverse)
library(multcomp)
library(multcompView)
library(cowplot)
library(openxlsx) # to read in .xlsx files

## Read in data

d <- NULL  

for (f in dir(".//Data/Light curves/lights/",
              pattern=".xlsx", recursive=T))
{
  print(f)
  tmp <- read.xlsx(paste(".//Data/Light curves/lights/",f,sep=""),
                   sheet=1,rows=c(17,19:30))
  g <- substr(f,1,nchar(f)-5)
  Ind <- strsplit(g, "_")[[1]]
  Ind1 <- strsplit(Ind,"-")[[1]]
  Period <- Ind1[4]
  Date <- paste(Ind1[3],Ind1[2],Ind1[1],sep="")
  Licor <- Ind[2]
  PlantNumber <- Ind[4]
  if(as.numeric(PlantNumber)<=12) {
    Regime = "Sinusoidal"
  } else {
    Regime = "Assymmetric"
  }
  colnames(tmp)[colnames(tmp) == "Fv/Fm"] <- "FvFm"
  colnames(tmp)[colnames(tmp) == "Fo'"] <- "Fop"
  PARi <- as.factor(c(1300,1100,900,700,550,400,250,150,100,50,0,0))
  d <- rbind(d, data.frame(Id=g, PlantNumber = PlantNumber, Licor=Licor, 
                           Regime = Regime, Period = Period, 
                           PARi=as.numeric(as.character(PARi)), 
                           Cond=tmp$gs, 
                           Photo=tmp$A,
                           Fo=tmp$Fo,
                           Fv=tmp$Fv,
                           Fm=tmp$Fm,
                           Fs=tmp$Fs,
                           Fmp=tmp$Fmp,
                           PhiPSII=tmp$PhiPS2,
                           FvFm=tmp$FvFm,
                           Fop=tmp$Fop
                           ))
}

str(d)
d$Period <- as.factor(d$Period)

## Make the final Fm value the one used across the measurements
df <- NULL
for(j in unique(d$Id)) 
  {
  temp <- d %>%
    filter(Id==j)
  print(temp$Id)
  temp$Fm <- last(temp$Fm)
  df <- rbind(df,temp)
}
    
df$NPQ <- (df$Fm-df$Fmp)/df$Fmp

#### Light Curves ####

d.A <- d %>%
  group_by(Regime,Period,PlantNumber) %>%
  slice(1:11) %>%
  group_by(Regime,Period,PARi) %>%
  summarise(N=n(),Mean=mean(Photo),SE=sd(Photo)/sqrt(N),.groups="drop")


a <- ggplot(d.A,aes(x=as.numeric(as.character(PARi)),y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Regime))+
  geom_point(size=2)+
  geom_errorbar()+
  ggtitle("A-Q Curve") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  facet_wrap(~Period)

plot(a)
save_plot(".//Outputs/A-Q Curves/A-Q.png",
          a,base_width=8)


## A-Q curve combined for all time periods and both treatments
d.A1 <- d %>%
  group_by(PlantNumber) %>%
  slice(1:11) %>%
  group_by(PARi) %>%
  summarise(N=n(),
            Mean=mean(Photo),
            SE=sd(Photo)/sqrt(N))

## Michaelis Menten kinetics
mm <- NULL
dd1 <- NULL

for (i in unique(d$Id))
{
  dd <- d %>%
    filter(d$Id == i)

  if(dd$Id[1]=="2020-09-08-1100_944_basil_13") {
    dd <- slice(dd,2:11)
  }
  
  if(dd$Id[1]=="2020-09-09-0800_944_basil_13") {
    dd <- slice(dd,2:11)
  }  
  if(dd$Id[1]=="2020-09-09-1100_500_basil_5") {
    dd <- slice(dd,2:11)
  }
  if(dd$Id[1]=="2020-09-09-1100_648_basil_16") {
    dd <- slice(dd,2:11)
  }
  if(dd$Id[1]=="2020-09-14-0800_500_basil_8") {
    dd <- slice(dd,2:11)
  }
  
  gmod <- function(p, PAR)
  {
    gmod <- (p[1]*PAR)/(p[2]+PAR)-p[3]
    # p[1] = Vmax, PAR = [Substrate], 
    # p[2] = Km,
    # p[3] = Rd 
    # The Michaelis-Menten model
    
    return(gmod)
  }
  
  opt <- function(p, gobs, PAR) # Mean square minimising function
  {
    return(mean((gmod(p, PAR)-gobs)^2))
  }
  
  init <- c(max(dd$Photo), 30, 0)
  out <- optim(init, opt, NULL, gobs=dd$Photo, 
               PAR=as.numeric(dd$PARi), control=list(maxit=5000))
  p <- out$par
  pto <- NULL
  for(n in 1:length(dd$PARi)) {
      
      ptoa <- (gmod(p,dd$PARi[n]))
      pto <- rbind(pto,ptoa)
  }
  dd$model <- pto
  
  # initial slope function
  is <- lm(Photo~PARi,data=filter(dd,PARi<200))
  
  int_i <- is[[1]][1]
  sl_i <- is[[1]][2]
  
  output <- ggplot(dd,aes(y=Photo,x=as.numeric(PARi), size=3, main=Id)) +
    geom_point() +
    geom_line(aes(y=model,x=as.numeric(PARi)),colour="red",size=2) +
    theme(legend.position="none") +
    ggtitle(paste("Plant #",dd$PlantNumber[1]," Time:",dd$Period[1],
                  dd$Regime[1]," Wave"))+
    geom_abline(slope=sl_i,intercept=int_i)
  output
  
  mm <- rbind(mm, data.frame(Id=i, 
                             Regime=dd$Regime[1], 
                             Period=dd$Period[1],
                             Vmax=p[1], 
                             Km=p[2], 
                             Rd=p[3],
                             Sl_init=sl_i,
                             int=int_i))
  dd1 <- rbind(dd1,data.frame(Id=i,
                              Regime=dd$Regime[1],
                              Period=dd$Period,
                              PAR=dd$PARi,
                              Cond=dd$Cond,
                              Photo=dd$Photo,
                              PhiII=dd$PhiPSII,
                              Model=dd$model
                              ))
  
  #ggsave(paste(".//Outputs/A-Q Curves/Graph_MM_model/",
  #             i,".png", sep=""))
  
}

## Initial slope
mm$slope_i <- (mm$Vmax+mm$Rd)/mm$Km

mm1 <- mm %>%
  #group_by(Regime,Period) %>%
  summarise(Rd=mean(Rd),
            Slope=mean(slope_i),
            Vmax=mean(Vmax),
            Km_se=sd(Km)/sqrt(n()),
            Km=mean(Km),
            Init_slope=mean(sl_i),
            Init_int=mean(int_i),
            .groups="keep")
mm1

dd2 <- dd1 %>%
  group_by(Regime,Period,PAR) %>%
  summarise(N=n(),Mean=mean(Photo),SE=sd(Photo)/sqrt(n),.groups="drop")
dd2$Regime <- as.factor(dd2$Regime)
dd2$Regime <- factor(dd2$Regime,labels=c("Asymmetric","Sinusoidal"))
str(dd2)

dd3 <- dd2 %>%
  filter(PAR==1300) %>%
  summarise(Mean=mean(Mean))
capture.output(dd3,file="./Outputs/Final Plots/Stats/Constants-A.txt")

a1 <- ggplot(dd1, aes(x=as.numeric(as.character(PAR)),y=Model)) +
  geom_point(size=1,position="jitter") +
  theme_bw()+
  theme(panel.grid=element_blank(),
        panel.border=element_rect(color = "black", 
                                  fill = NA, 
                                  linewidth = 1),
        axis.line = element_line(colour = "black"),
        axis.title=element_text(size=11),
        legend.position = "right")+
  xlab(expression(paste("PFD (",~mu,mol,~m^-2,~s^-1," )")))+
  ylab(expression(paste("Assimilation (",~mu,mol,~CO[2],~m^-~2,~s^-~1," )"))) +
  annotate("segment",x=mean(mm$Km),xend=mean(mm$Km),
           y=-4.5,yend=mean(mm$Vmax/2)-mean(mm$Rd),colour="darkgreen")+
  annotate("segment",x=-85, xend=mean(mm$Km), y=mean(mm$Vmax/2)-mean(mm$Rd),
           yend=mean(mm$Vmax/2)-mean(mm$Rd),colour="darkgreen") +
  # min-max values at 100 and 450 PPFD
  annotate("segment",x=100, xend=100, y=-4.5, 
           yend=mean(mm$Vmax)*100/(mean(mm$Km)+100)-mean(mm$Rd), colour="darkgreen",
           linetype="dashed") +
  annotate("segment",x=-85, xend=100, 
           y=mean(mm$Vmax)*100/(mean(mm$Km)+100)-mean(mm$Rd),
           yend=mean(mm$Vmax)*100/(mean(mm$Km)+100)-mean(mm$Rd), 
               colour="darkgreen", linetype="dashed") +
  annotate("segment",x=450, xend=450, # 1/2 Vmax and Km
                   y=-4.5, yend=mean(mm$Vmax)*450/(mean(mm$Km)+450)-mean(mm$Rd), 
               colour="darkgreen",linetype="dashed") +
  annotate("segment", x=-85, 
           xend=450, y=mean(mm$Vmax)*450/(mean(mm$Km)+450)-mean(mm$Rd),
           yend=mean(mm$Vmax)*450/(mean(mm$Km)+450)-mean(mm$Rd), 
           colour="darkgreen",linetype="dashed") +
  # add some text 
  annotate("text",x=mean(mm$Km)-40, y=6,
           label=paste("Km = \n", round(mean(mm$Km),0)), size=3)+
  annotate("text",x=60,y=mean(mm$Vmax/2)-1.5*mean(mm$Rd),
           label="1/2 Vmax",size=3, hjust=0)+
  stat_function(fun = function(x) mean(mm$Vmax)*x/(mean(mm$Km)+x)-mean(mm$Rd),
                colour="red",size=1)+
  annotate("text",x=470,y=0,label="PFD =\n 450",size=3, hjust=0)+
  annotate("text",x=110,y=0,label="PFD =\n 100",size=3, hjust=0) +
  coord_cartesian(clip = 'on', ylim=c(-3, 28),xlim=c(-20,1320)) +
  scale_x_continuous(breaks = c(0,250,500,750,1000,1250)) 
  
#scale_y_continuous(limits=c(-5,28))
  #scale_y_continuous(expand = expansion(add = c(0.05, 0.5))) 

a1

### THIS IS THE KEY PLOT FROM THESE DATA FOR FIG 1 OF THE PAPER ###
save_plot(".//Outputs/A-Q Curves/A-Q_combined.png",
          a1,base_width=8)
save(a1,file="./Outputs/Intensity_min_max_selection.rdata")
###-------------------------------------------------------------###

# NPQ vs PAR #
#------------#

# modelling NPQ using Gomperz function
# can be used to check midpoint and max slope of rise

# create function
GOMP <- NULL

for (id in unique(df$Id))
{
  
  print(id)
  AQtmp <- df %>%
    filter(Id==id) %>%
    slice(1:10)
  
  if(AQtmp$Id[1]=="2020-09-08-1100_944_basil_13") {
    AQtmp <- slice(AQtmp,2:10)
  }
  
  if(AQtmp$Id[1]=="2020-09-09-0800_944_basil_13") {
  AQtmp <- slice(AQtmp,2:10)
}  
 if(AQtmp$Id[1]=="2020-09-09-1100_500_basil_5") {
   AQtmp <- slice(AQtmp,2:10)
 }
  if(AQtmp$Id[1]=="2020-09-09-1100_648_basil_16") {
    AQtmp <- slice(AQtmp,2:10)
  }
  if(AQtmp$Id[1]=="2020-09-14-0800_500_basil_8") {
    AQtmp <- slice(AQtmp,2:10)
  }
  
  gmod <- function(p, PAR)
  {
    gmod <- p[1]*exp(-p[2]*exp(-p[3]*PAR))
    #p[1] = a, p[2] = b, p[3] = c
    # a = max or min value; b slides the curve along the x axis; c = curvature
    
    return(gmod)
  }
  
  opt <- function(p, gobs, PAR) # Mean square minimising function
  {
    return(mean((gmod(p, PAR)-gobs)^2))
  }
  
  init <- c(max(AQtmp$NPQ), 5, 0.0003)
  out <- optim(init, opt, NULL, gobs=AQtmp$NPQ, 
               PAR=as.numeric(AQtmp$PARi), control=list(maxit=5000))
  p <- out$par
  pto <- NULL
  for(n in 1:length(AQtmp$PARi)) {
    
    ptoa <- (gmod(p,as.numeric(AQtmp$PARi[n])))
    pto <- rbind(pto,ptoa)
  }
  AQtmp$model <- pto
  output <- ggplot(AQtmp,aes(y=NPQ,x=as.numeric(PARi), size=3, main=id)) +
    geom_point() +
    geom_line(aes(y=model,x=as.numeric(PARi)),colour="red",size=2) +
    theme(legend.position="none") +
    ggtitle(paste("Plant #",AQtmp$PlantNumber[1]," Time:",AQtmp$Period[1],
                  AQtmp$Regime[1]," Wave"))
  print(output)
  
  GOMP <- rbind(GOMP, data.frame(Id=id, Regime=AQtmp$Regime[1], Period=AQtmp$Period[1],
                                 a=p[1], b=p[2], c=p[3]))
  
  ggsave(paste(".//Outputs/A-Q Curves/Graph_Gomperz/",
                 id,".png", sep=""))
  
}

GOMP$Mid <- -log(log(2)/GOMP$b)/GOMP$c # midpoint of NPQ curve
GOMP$Max <- log(GOMP$b/GOMP$c) # max slope of NPQ curve

GOM <- GOMP %>%
  group_by(Regime,Period) %>%
  summarise(N=n(),Amean=mean(a),Bmean=mean(b),
            Cmean=mean(c),Midmean=mean(Mid),Maxmean=mean(Max),
            .groups="drop")

AQ.NPQ <- df %>%
  group_by(Period,PARi,Regime) %>%
  summarise(N=n(),Mean=mean(NPQ),SE=sd(NPQ)/sqrt(N),.groups="drop")

am <- merge(cbind(AQ.NPQ, X=rownames(AQ.NPQ)), cbind(GOM, variable=rownames(GOM)))

am1 <- am %>% unite("id",Period:Regime,sep="_")
am1$Period <- am$Period
am1$Regime <- am$Regime
am1$PARi <- as.numeric(as.character(am1$PARi))

df1 <- NULL

for (idd in unique(am1$id))
{
  
  print(idd)
  am1tmp <- filter(am1,id==idd)
  
  gmod <- function(p, PARi)
  {
    gmod <- p[1]*exp(-p[2]*exp(-p[3]*as.numeric(as.character(PARi))))
    #p[1] = a, p[2] = b, p[3] = c
    # a = max or min value; b slides the curve along the x axis; c = curvature
    
    return(gmod)
  }
  
  opt <- function(p, gobs, PARi) # Mean square minimising function
  {
    return(mean((gmod(p, PARi)-gobs)^2))
  }
  
  init <- c(max(am1tmp$Mean), 5, 0.0003)
  out <- optim(init, opt, NULL, gobs=am1tmp$Mean, PARi=am1tmp$PARi, control=list(maxit=2000))
  p <- out$par
  pto <- NULL
  for(n in 1:length(am1tmp$PARi)) {
    ptoa <- (gmod(p,am1tmp$PARi[n]))
    pto <- rbind(pto,ptoa)
  }
  am1tmp$model <- pto
  df1 <- rbind(df1,data.frame(id=am1tmp$id[1],
                              PARi=am1tmp$PARi,
                              Mean=am1tmp$Mean,
                              SE=am1tmp$SE,
                              Period=am1tmp$Period[1],
                              Regime=am1tmp$Regime[1],
                              model=am1tmp$model))
  
}

b <- ggplot(df1,aes(x=PARi,y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Period))+
  geom_point(size=2) +
  geom_errorbar()+
  geom_line(aes(x=as.numeric(as.character(PARi)),y=model))+
  ggtitle("NPQ-Q Curve")+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  ylab("NPQ")+
  xlab(expression(paste("PAR ( ",mu,mol,~m^-2,~s^-1," )")))+
  facet_wrap(~Regime)

plot(b)

## qP

d$qP <- (d$Fmp-d$Fs)/(d$Fmp-d$Fop)

AQ.qP <- d %>%
  group_by(Regime,Period,PARi) %>%
  summarise(N=n(),Mean=mean(qP),SE=sd(qP)/sqrt(N),.groups="drop")

c <- ggplot(AQ.qP,aes(x=as.numeric(as.character(PARi)),y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Period))+
  geom_point(size=3)+
  geom_errorbar()+
  ggtitle("qP-Q curve") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  ylim(0.3,1) +
  facet_wrap(~Regime)

plot(c)

## qP comps ##
#------------#
qP <- d %>%
  filter(PARi==400) 

qPmod <- lm(qP~Regime*Period,data=qP)
summary(qPmod)

## Fv Fm comps
fv <- d %>%
  group_by(Regime,Period,PlantNumber) %>%
  slice(12) #%>%
  #na.omit()

fvmod <- lm(FvFm~Regime*Period,data=fv)
summary(fvmod) # sig

fvfm <- d %>%
  group_by(Regime,Period,PlantNumber) %>%
  slice(12) %>%
  filter(!is.na(FvFm)) %>%
  group_by(Regime) %>%
  summarise(N=n(),Mean=mean(FvFm),SE=sd(FvFm)/sqrt(N))

dd <- ggplot(fvfm,aes(x=Regime,y=Mean,ymax=Mean+SE,ymin=Mean-SE,fill=Regime))+
  geom_col()+
  geom_errorbar()+
  ggtitle("Fv / Fm")+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  ylim(0,1)

plot(dd)

## NPQ analysis ##
#----------------#
mod.mid <- lm(Mid~Regime,data=GOMP)
summary(mod.mid) # nearly sig

mod.max <- lm(Max~Regime,data=GOMP)
summary(mod.max) # sig

GOMP1 <- GOMP %>%
  group_by(Regime,Period) %>%
  summarise(N=n(),Midmean=mean(Mid),MidSE=sd(Mid)/sqrt(N),
            Maxmean=mean(Max),MaxSE=sd(Max)/sqrt(N),.groups="drop")

e <- ggplot(GOMP1,aes(x=Period,y=Midmean,ymin=Midmean-MidSE,ymax=Midmean+MidSE,fill=Period))+
  geom_col()+
  geom_errorbar()+
  ylab(expression(paste("Midpoint of curve ( PAR, ",~mu,mol,~m^-2,~s^-1," )")))+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  facet_wrap(~Regime)
plot(e)

g <- ggplot(GOMP1,aes(x=Period,y=Maxmean,ymin=Maxmean-MaxSE,ymax=Maxmean+MaxSE,fill=Period))+
  geom_col()+
  geom_errorbar()+
  ggtitle("Max slope of NPQ curve") +
  ylab(expression(paste("Max slope of curve ( ",~mu,mol,~m^-2,~s^-1,quantum^-1," )")))+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  facet_wrap(~Regime)
plot(g)

## Fq' / Fm' analysis ##
#----------------------#
fqmp <- d %>%
  filter(PARi!=0) %>%
  group_by(Regime,Period,PlantNumber) %>%
  #slice(1:12) %>%
  mutate(Fqmp=(Fmp-Fs)/Fmp)

fqmpmod <- lm(Fqmp~Regime*Period*PARi,data=fqmp)
summary(fqmpmod)

fqmp.A <- fqmp %>%
  group_by(Regime,Period,PARi) %>%
  summarise(N=n(),Mean=mean(Fqmp),SE=sd(Fqmp)/sqrt(N),.groups="drop")


i <- ggplot(fqmp.A,aes(x=as.numeric(as.character(PARi)),y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Period))+
  geom_point(size=3)+
  geom_errorbar()+
  ggtitle("Fq' / Fm' ") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  facet_wrap(~Regime)

plot(i)

## Block out Fq'/Fm' for 100 to 450 PAR and note bounds for Setpoint
sumup <- fqmp.A %>%
  group_by(as.factor(PARi)) %>%
  summarise(Mean=mean(Mean), SE=mean(SE))
sumup <- sumup %>%
  rename(PARi='as.factor(PARi)')

# analyse as s-curve

fqpplot <- ggplot(sumup,aes(x=as.factor(PARi),y=Mean,
                            ymin=Mean-SE,ymax=Mean+SE))+
  geom_point()
fqpplot



## Fv' / Fm' analysis by estimation in the absence of Fo' values ##
#-----------------------------------------------------------------#


ddd <- d %>%
  group_by(Id,Period,PlantNumber,Regime,PARi) %>%
  slice(1:10) %>%
  #select(Period,PlantNumber,Regime,PARi,Fop,Fmp) %>%
  mutate(FvFmp=(Fmp-Fop)/(Fmp) )

fvmpmod <- lm(FvFmp~Regime*Period*PARi,data=ddd)
summary(fvmpmod)
# No effect of time, effect of light intensity, no effect of wave type

fvmp.A <- ddd %>%
  group_by(Period,Regime,PARi) %>%
  summarise(N=n(),Mean=mean(FvFmp),SE=sd(FvFmp)/sqrt(N),.groups="drop")


h <- ggplot(fvmp.A,aes(x=as.numeric(as.character(PARi)),y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Period))+
  geom_point(size=3)+
  geom_errorbar()+
  ggtitle("Fv' / Fm' ") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) +
  xlab(expression(paste("PAR ( ",~mu,mol,~m^-2,s^-1," )")))+
  facet_wrap(~Regime)

plot(h)

## Initial gs at prervailing Q
gs <- read.xlsx(".//Data/Light curves/gs/All gs.xlsx",
              sheet=1,cols=c(1:3,7,9:15))
gs$Period <- as.factor(gs$Time)
gs$Period <- factor(gs$Period,labels=c("0600","0900","1200","1500"))
gs$Regime <- as.factor(gs$Treatment)

gs1 <- gs %>%
  group_by(Period,Regime) %>%
  summarise(N=n(),Mean=mean(gsw),SE=sd(gsw)/sqrt(N),.groups="drop")

gsmod <- lm(gsw~Regime*Period,data=gs)
summary(gsmod)

j <- ggplot(gs1,aes(x=Period,y=Mean,ymax=Mean+SE,ymin=Mean-SE,fill=Period))+
  geom_col(position=position_dodge(width=20))+
  geom_errorbar(position=position_dodge(width=20))+
  ggtitle("Conductance snapshot pre-start ") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) +
  #ylim(0,0.35)+
  ylab(expression(paste(italic(g[s])," ( ",~mol,~m^-2,~s^-1," )")))+
  facet_wrap(~Regime)
plot(j)

f <- plot_grid(j+theme(legend.position="none"), # pre-start gs snapshot
               a+theme(legend.position="none"), # A-Q curve
               c+theme(legend.position="none"), # qP-Q (Fq' / Fv')
               i+theme(legend.position="none"), #Fq' / Fm'
               h+theme(legend.position="none"), # Fv ' / Fm'
               dd+theme(legend.position="none"), # Fv / Fm
               b+theme(legend.position="bottom"), # NPQ-Q
               e+theme(legend.position="none"), # NPQ midpoint
               g+theme(legend.position="none"), # NPQ max slope
               align="vh",
               labels="auto",
               nrow=3)  

plot(f)
save_plot(".//Outputs/Combined Sin vs Ass Light Response Plot.png",f,base_height=12)

